from pyparsing import Literal

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector, ObjectMethodInvocation
from codeable_detectors.evidences import FailedEvidence, ServiceEvidence, LinkEvidence
from codeable_detectors.go.go_detectors import detect_go_import
from codeable_detectors.pyparsing_patterns import ID, round_braces_block
from codeable_detectors.utils import update_keyword_args


class GoAMQPService(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["go"]

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        is_detected = False

        import_matches = detect_go_import(ctx, "github.com/streadway/amqp")
        if not import_matches:
            return FailedEvidence("go amqp import not found")

        for import_match in import_matches:
            matches.append(import_match)
            alias = "amqp"
            if "alias" in import_match.kwargs:
                alias = import_match.kwargs["alias"]
            # check if a connection is established with Dial
            connection_dial_matches = ctx.matches_pattern(Literal(alias) + Literal(".") + Literal("Dial") +
                                                          round_braces_block)
            if connection_dial_matches:
                matches.extend(connection_dial_matches)
                is_detected = True

        if is_detected:
            return ServiceEvidence(matches).set_properties(detector_technology_types=["go", "amqp"], kwargs=kwargs)
        return FailedEvidence("no amqp Dial call found")


class GoAMQPMessageConsumerLink(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["go"]
        self.consume_invocation_detector = ObjectMethodInvocation(ID, "Consume", "go", closing_semicolon=False)

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        options = update_keyword_args({'channel': None}, kwargs)
        tagged_values = {}
        if options["channel"]:
            tagged_values = {"channel": options["channel"]}

        import_matches = detect_go_import(ctx, "github.com/streadway/amqp")
        if not import_matches:
            return FailedEvidence("go amqp import not found")
        matches.extend(import_matches)

        evidence = self.consume_invocation_detector.detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return FailedEvidence("cannot find 'Consume' invocation")
        matches.extend(evidence.matches)

        return LinkEvidence(matches).set_properties(detector_link_types=["messageConsumer"],
                                                    detector_technology_types=["go", "amqp"],
                                                    detector_tagged_values=tagged_values, kwargs=kwargs)
